/*
 * Decompiled with CFR 0.152.
 */
package info.oais.oaisif.packagevalidator;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import info.oais.oaisif.packagevalidator.PackageValidator;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Pattern;

public class PackageValidator {
    private List<ValidationError> errors = new ArrayList();
    private Set<String> knownInfoObjectIDs = new HashSet();
    private Set<String> knownRepInfoIDs = new HashSet();
    private static final String[] VALID_PACKAGE_TYPES = new String[]{"AIP", "General", "InfoObjectRequest", "Query", "QueryResponse", "ObjectRequestResponse", "ErrorResponse"};
    private static final String[] VALID_IDENTIFIER_TYPES = new String[]{"URI", "URL", "URN", "DOI", "ARK", "HANDLE", "PURL", "UUID", "JSONPath", "PhysicalLocation", "OTHER"};
    private static final String[] VALID_REFERENCE_TYPES = new String[]{"INTERNAL_INFOOBJECT", "INTERNAL_REPINFO", "REMOTE_INFOOBJECT", "REMOTE_REPINFO"};
    private static final String[] VALID_ENCODING_TYPES = new String[]{"ASCII7", "UTF8", "UTF16", "BASE64", "BASE32", "HEX", "OTHER"};
    private static final Pattern VERSION_PATTERN = Pattern.compile("^\\d+\\.\\d+\\.\\d+$");
    private static final Pattern REPINFO_CATEGORY_PATTERN = Pattern.compile("^(Structure|Semantic|Other|Combined)(/[A-Za-z]+)*$");

    public boolean validateFile(String filePath) throws IOException {
        ObjectMapper mapper = new ObjectMapper();
        JsonNode json = mapper.readTree(new File(filePath));
        this.knownInfoObjectIDs.clear();
        this.knownRepInfoIDs.clear();
        this.validateRoot(json);
        return this.errors.isEmpty();
    }

    private void validateRoot(JsonNode json) {
        if (!json.has("InformationPackage")) {
            this.errors.add(new ValidationError("$.InformationPackage", "Root object must have an 'InformationPackage' property", json, "Missing 'InformationPackage' property"));
        } else {
            this.validateInformationPackage(json.get("InformationPackage"), "$.InformationPackage");
        }
    }

    private void validateInformationPackage(JsonNode ip, String path) {
        String version;
        String[] requiredFields;
        String[] stringArray = requiredFields = new String[]{"InformationObject", "IsDeclaredComplete", "PackageDescription", "PackageType", "version"};
        int n = requiredFields.length;
        int n2 = 0;
        while (n2 < n) {
            String field = stringArray[n2];
            if (!ip.has(field)) {
                this.errors.add(new ValidationError(path + "." + field, "InformationPackage must have a '" + field + "' property", ip, "Missing '" + field + "' property"));
            }
            ++n2;
        }
        if (ip.has("version") && !VERSION_PATTERN.matcher(version = ip.get("version").asText()).matches()) {
            this.errors.add(new ValidationError(path + ".version", "Invalid version format: " + version, ip, "Invalid version format"));
        }
        if (ip.has("PackageType")) {
            String packageType = ip.get("PackageType").asText();
            if (packageType.trim().isEmpty()) {
                this.errors.add(new ValidationError(path + ".PackageType", "PackageType cannot be empty", ip, "Empty PackageType"));
            } else {
                this.validateEnum(packageType, VALID_PACKAGE_TYPES, "PackageType", path + ".PackageType");
            }
        }
        if (ip.has("IsDeclaredComplete") && ip.has("PackageType")) {
            boolean isDeclaredComplete = ip.get("IsDeclaredComplete").asBoolean();
            if ("AIP".equals(ip.get("PackageType").asText()) && isDeclaredComplete && !ip.has("PDI")) {
                this.errors.add(new ValidationError(path + ".PDI", "An AIP with IsDeclaredComplete=true must have a PDI property", ip, "Missing PDI property"));
            }
        }
        if (ip.has("InformationObject")) {
            this.validateInformationObject(ip.get("InformationObject"), path + ".InformationObject");
        }
    }

    void validateEnum(String value, String[] validValues, String fieldName, String path) {
        String[] stringArray = validValues;
        int n = validValues.length;
        int n2 = 0;
        while (n2 < n) {
            String validValue = stringArray[n2];
            if (validValue.equals(value)) {
                return;
            }
            ++n2;
        }
        this.errors.add(new ValidationError(path, "Invalid " + fieldName + ": " + value, null, "Invalid value"));
    }

    private void validateInformationObject(JsonNode io, String path) {
        if (io == null || !io.isObject()) {
            this.errors.add(new ValidationError(path, "InformationObject must be a JSON object", io, "Invalid InformationObject format"));
            return;
        }
        if (io.has("InfoObjectID")) {
            String infoObjectID = io.get("InfoObjectID").asText();
            if (infoObjectID.trim().isEmpty()) {
                this.errors.add(new ValidationError(path + ".InfoObjectID", "InfoObjectID cannot be empty", io, "Empty InfoObjectID"));
            }
            this.knownInfoObjectIDs.add(infoObjectID);
        }
        boolean hasDataObjectAndRepInfo = io.has("DataObject") && io.has("RepresentationInformation");
        boolean hasIdentifierObject = io.has("IdentifierObject");
        if (!hasDataObjectAndRepInfo && !hasIdentifierObject) {
            this.errors.add(new ValidationError(path, "InformationObject must have either (DataObject and RepresentationInformation) or IdentifierObject", io, "Missing required properties"));
        }
        if (hasDataObjectAndRepInfo && hasIdentifierObject) {
            this.errors.add(new ValidationError(path, "InformationObject cannot have both (DataObject and RepresentationInformation) and IdentifierObject", io, "Conflicting properties"));
        }
        if (hasDataObjectAndRepInfo) {
            this.validateDataObject(io.get("DataObject"), path + ".DataObject");
            this.validateRepresentationInformation(io.get("RepresentationInformation"), path + ".RepresentationInformation");
        }
        if (hasIdentifierObject) {
            this.validateIdentifierObject(io.get("IdentifierObject"), path + ".IdentifierObject");
        }
    }

    private void validateDataObject(JsonNode dataObj, String path) {
        String size;
        if (dataObj == null || !dataObj.isObject()) {
            this.errors.add(new ValidationError(path, "DataObject must be a JSON object", dataObj, "Invalid DataObject format"));
            return;
        }
        boolean hasEncodedObject = dataObj.has("EncodedObject");
        boolean hasIdentifierObject = dataObj.has("IdentifierObject");
        if (!hasEncodedObject && !hasIdentifierObject) {
            this.errors.add(new ValidationError(path, "DataObject must have either EncodedObject or IdentifierObject", dataObj, "Missing required properties"));
        }
        if (hasEncodedObject && hasIdentifierObject) {
            this.errors.add(new ValidationError(path, "DataObject cannot have both EncodedObject and IdentifierObject", dataObj, "Conflicting properties"));
        }
        if (hasEncodedObject) {
            this.validateEncodedObject(dataObj.get("EncodedObject"), path + ".EncodedObject");
        }
        if (hasIdentifierObject) {
            this.validateIdentifierObject(dataObj.get("IdentifierObject"), path + ".IdentifierObject");
        }
        if (dataObj.has("size") && !(size = dataObj.get("size").asText()).matches("^[0-9]+$")) {
            this.errors.add(new ValidationError(path + ".size", "DataObject size must be a non-negative integer as a string", dataObj, "Invalid size format"));
        }
    }

    private void validateEncodedObject(JsonNode encodedObj, String path) {
        if (encodedObj == null || !encodedObj.isObject()) {
            this.errors.add(new ValidationError(path, "EncodedObject must be a JSON object", encodedObj, "Invalid EncodedObject format"));
            return;
        }
        if (!encodedObj.has("Encoding")) {
            this.errors.add(new ValidationError(path + ".Encoding", "EncodedObject must have an 'Encoding' property", encodedObj, "Missing 'Encoding' property"));
        } else {
            String encoding = encodedObj.get("Encoding").asText();
            this.validateEnum(encoding, VALID_ENCODING_TYPES, "Encoding", path + ".Encoding");
            if ("OTHER".equals(encoding) && !encodedObj.has("CustomEncoding")) {
                this.errors.add(new ValidationError(path + ".CustomEncoding", "EncodedObject with Encoding='OTHER' must have a 'CustomEncoding' property", encodedObj, "Missing 'CustomEncoding' property"));
            }
        }
        if (!encodedObj.has("EncodedContent")) {
            this.errors.add(new ValidationError(path + ".EncodedContent", "EncodedObject must have an 'EncodedContent' property", encodedObj, "Missing 'EncodedContent' property"));
        } else {
            String encodedContent = encodedObj.get("EncodedContent").asText();
            if (encodedContent.trim().isEmpty()) {
                this.errors.add(new ValidationError(path + ".EncodedContent", "EncodedContent cannot be empty", encodedObj, "Empty 'EncodedContent'"));
            }
        }
    }

    private void validateIdentifierObject(JsonNode idObj, String path) {
        String description;
        String version;
        if (idObj == null || !idObj.isObject()) {
            this.errors.add(new ValidationError(path, "IdentifierObject must be a JSON object", idObj, "Invalid IdentifierObject format"));
            return;
        }
        if (!idObj.has("IdentifierString")) {
            this.errors.add(new ValidationError(path + ".IdentifierString", "IdentifierObject must have an 'IdentifierString' property", idObj, "Missing 'IdentifierString' property"));
        } else {
            String identifierString = idObj.get("IdentifierString").asText();
            if (identifierString.trim().isEmpty()) {
                this.errors.add(new ValidationError(path + ".IdentifierString", "IdentifierString cannot be empty", idObj, "Empty 'IdentifierString'"));
            }
        }
        if (!idObj.has("IdentifierType")) {
            this.errors.add(new ValidationError(path + ".IdentifierType", "IdentifierObject must have an 'IdentifierType' property", idObj, "Missing 'IdentifierType' property"));
        } else {
            String identifierType = idObj.get("IdentifierType").asText();
            this.validateEnum(identifierType, VALID_IDENTIFIER_TYPES, "IdentifierType", path + ".IdentifierType");
            if ("OTHER".equals(identifierType) && !idObj.has("CustomIdentifierType")) {
                this.errors.add(new ValidationError(path + ".CustomIdentifierType", "IdentifierObject with IdentifierType='OTHER' must have a 'CustomIdentifierType' property", idObj, "Missing 'CustomIdentifierType' property"));
            }
        }
        if (idObj.has("version") && !VERSION_PATTERN.matcher(version = idObj.get("version").asText()).matches()) {
            this.errors.add(new ValidationError(path + ".version", "Invalid version format: " + version, idObj, "Invalid version format"));
        }
        if (idObj.has("Description") && (description = idObj.get("Description").asText()).trim().isEmpty()) {
            this.errors.add(new ValidationError(path + ".Description", "Description cannot be empty", idObj, "Empty 'Description'"));
        }
        if (idObj.has("ReferenceType")) {
            String referenceType = idObj.get("ReferenceType").asText();
            this.validateEnum(referenceType, VALID_REFERENCE_TYPES, "ReferenceType", path + ".ReferenceType");
        }
    }

    private void validateRepresentationInformation(JsonNode repInfo, String path) {
        String recursionHalt;
        if (repInfo == null || !repInfo.isObject()) {
            this.errors.add(new ValidationError(path, "RepresentationInformation must be a JSON object", repInfo, "Invalid RepresentationInformation format"));
            return;
        }
        if (repInfo.has("RepInfoID")) {
            String repInfoID = repInfo.get("RepInfoID").asText();
            if (repInfoID.trim().isEmpty()) {
                this.errors.add(new ValidationError(path + ".RepInfoID", "RepInfoID cannot be empty", repInfo, "Empty 'RepInfoID'"));
            }
            this.knownRepInfoIDs.add(repInfoID);
        }
        if (repInfo.has("RepInfoCategory")) {
            String repInfoCategory = repInfo.get("RepInfoCategory").asText();
            if (!REPINFO_CATEGORY_PATTERN.matcher(repInfoCategory).matches()) {
                this.errors.add(new ValidationError(path + ".RepInfoCategory", "Invalid RepInfoCategory format: " + repInfoCategory, repInfo, "Invalid 'RepInfoCategory' format"));
            }
        } else {
            this.errors.add(new ValidationError(path + ".RepInfoCategory", "RepresentationInformation must have a 'RepInfoCategory' property", repInfo, "Missing 'RepInfoCategory' property"));
        }
        boolean hasAndGroup = repInfo.has("AndGroup");
        boolean hasOrGroup = repInfo.has("OrGroup");
        boolean hasInformationObject = repInfo.has("InformationObject");
        boolean hasRecursionHalt = repInfo.has("RecursionHalt");
        if (!(hasAndGroup || hasOrGroup || hasInformationObject || hasRecursionHalt)) {
            this.errors.add(new ValidationError(path, "RepresentationInformation must have one of 'AndGroup', 'OrGroup', 'InformationObject', or 'RecursionHalt'", repInfo, "Missing required properties"));
        }
        if (hasAndGroup) {
            this.validateAndGroup(repInfo.get("AndGroup"), path + ".AndGroup");
        }
        if (hasOrGroup) {
            this.validateOrGroup(repInfo.get("OrGroup"), path + ".OrGroup");
        }
        if (hasInformationObject) {
            this.validateInformationObject(repInfo.get("InformationObject"), path + ".InformationObject");
        }
        if (hasRecursionHalt && (recursionHalt = repInfo.get("RecursionHalt").asText()).trim().isEmpty()) {
            this.errors.add(new ValidationError(path + ".RecursionHalt", "RecursionHalt cannot be empty", repInfo, "Empty 'RecursionHalt'"));
        }
    }

    private void validateAndGroup(JsonNode andGroup, String path) {
        if (!andGroup.isArray() || andGroup.size() < 1) {
            this.errors.add(new ValidationError(path, "AndGroup must contain at least one element", andGroup, "Empty 'AndGroup'"));
        }
        int i = 0;
        while (i < andGroup.size()) {
            this.validateRepresentationInformation(andGroup.get(i), path + "[" + i + "]");
            ++i;
        }
    }

    private void validateOrGroup(JsonNode orGroup, String path) {
        if (!orGroup.isArray() || orGroup.size() < 1) {
            this.errors.add(new ValidationError(path, "OrGroup must contain at least one element", orGroup, "Empty 'OrGroup'"));
        }
        int i = 0;
        while (i < orGroup.size()) {
            this.validateRepresentationInformation(orGroup.get(i), path + "[" + i + "]");
            ++i;
        }
    }

    public List<ValidationError> getErrors() {
        return this.errors;
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.out.println("Usage: java PackageValidator <jsonFilePath>");
            System.exit(1);
        }
        PackageValidator validator = new PackageValidator();
        try {
            boolean isValid = validator.validateFile(args[0]);
            System.out.println("Validation result: " + (isValid ? "VALID" : "INVALID"));
            if (!isValid) {
                int i = 0;
                for (ValidationError error : validator.getErrors()) {
                    System.out.println("Error [" + i++ + "] at " + error.toStringBrief() + "\n");
                }
            }
        }
        catch (IOException e) {
            System.out.println("File error: " + e.getMessage());
        }
    }
}

